/*
 * Decompiled with CFR 0.152.
 */
package com.university.bookstore.search;

import com.university.bookstore.model.Material;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.logging.Logger;

public class ModernSearchCache
implements AutoCloseable {
    private static final Logger LOGGER = Logger.getLogger(ModernSearchCache.class.getName());
    private final Map<String, CacheEntry> cache;
    private final ExecutorService loadingExecutor;
    private final ScheduledExecutorService maintenanceExecutor;
    private final int maxSize;
    private final Duration ttl;
    private final Duration idleTime;
    private final LongAdder hitCount = new LongAdder();
    private final LongAdder missCount = new LongAdder();
    private final LongAdder evictionCount = new LongAdder();
    private final LongAdder loadCount = new LongAdder();
    private final LongAdder totalLoadTime = new LongAdder();
    private volatile boolean closed = false;

    public ModernSearchCache() {
        this(1000, Duration.ofMinutes(10L), Duration.ofMinutes(5L));
    }

    public ModernSearchCache(int maxSize, Duration ttl, Duration idleTime) {
        this.maxSize = maxSize;
        this.ttl = ttl;
        this.idleTime = idleTime;
        this.cache = new ConcurrentHashMap<String, CacheEntry>();
        this.loadingExecutor = Executors.newCachedThreadPool(r -> {
            Thread t = new Thread(r, "Cache-Loader");
            t.setDaemon(true);
            return t;
        });
        this.maintenanceExecutor = Executors.newScheduledThreadPool(1, r -> {
            Thread t = new Thread(r, "Cache-Maintenance");
            t.setDaemon(true);
            return t;
        });
        this.scheduleMaintenance();
    }

    private void scheduleMaintenance() {
        this.maintenanceExecutor.scheduleWithFixedDelay(this::performMaintenance, 1L, 1L, TimeUnit.MINUTES);
        this.maintenanceExecutor.scheduleWithFixedDelay(() -> LOGGER.info(this.getStats().getSummary()), 5L, 5L, TimeUnit.MINUTES);
    }

    private void performMaintenance() {
        if (this.closed) {
            return;
        }
        int removed = 0;
        Iterator<Map.Entry<String, CacheEntry>> iterator = this.cache.entrySet().iterator();
        while (iterator.hasNext()) {
            Map.Entry<String, CacheEntry> entry = iterator.next();
            CacheEntry cacheEntry = entry.getValue();
            if (!cacheEntry.isExpired(this.ttl) && !cacheEntry.isStale(this.idleTime)) continue;
            iterator.remove();
            this.evictionCount.increment();
            ++removed;
        }
        if (removed > 0) {
            LOGGER.fine("Maintenance: Removed " + removed + " expired/stale entries");
        }
        if (this.cache.size() > this.maxSize) {
            this.performSizeBasedEviction();
        }
    }

    private void performSizeBasedEviction() {
        int toRemove = this.cache.size() - this.maxSize;
        if (toRemove <= 0) {
            return;
        }
        ArrayList<Map.Entry<String, CacheEntry>> entries = new ArrayList<Map.Entry<String, CacheEntry>>(this.cache.entrySet());
        entries.sort(Comparator.comparing(e -> ((CacheEntry)e.getValue()).lastAccessedAt));
        for (int i = 0; i < toRemove && i < entries.size(); ++i) {
            this.cache.remove(((Map.Entry)entries.get(i)).getKey());
            this.evictionCount.increment();
        }
        LOGGER.fine("Size-based eviction: Removed " + toRemove + " entries");
    }

    public List<Material> get(String key, Function<String, List<Material>> loader) {
        Objects.requireNonNull(key, "Key cannot be null");
        Objects.requireNonNull(loader, "Loader cannot be null");
        this.ensureNotClosed();
        CacheEntry entry = this.cache.get(key);
        if (entry != null && !entry.isExpired(this.ttl)) {
            this.hitCount.increment();
            this.cache.put(key, entry.recordAccess());
            return new ArrayList<Material>(entry.value);
        }
        this.missCount.increment();
        long startTime = System.currentTimeMillis();
        List<Material> value = loader.apply(key);
        long loadTime = System.currentTimeMillis() - startTime;
        this.loadCount.increment();
        this.totalLoadTime.add(loadTime);
        if (value != null && !value.isEmpty()) {
            this.put(key, value);
        }
        return value != null ? new ArrayList<Material>(value) : new ArrayList();
    }

    public CompletableFuture<List<Material>> getAsync(String key, Function<String, CompletableFuture<List<Material>>> asyncLoader) {
        Objects.requireNonNull(key, "Key cannot be null");
        Objects.requireNonNull(asyncLoader, "Async loader cannot be null");
        this.ensureNotClosed();
        CacheEntry entry = this.cache.get(key);
        if (entry != null && !entry.isExpired(this.ttl)) {
            this.hitCount.increment();
            this.cache.put(key, entry.recordAccess());
            return CompletableFuture.completedFuture(new ArrayList<Material>(entry.value));
        }
        this.missCount.increment();
        long startTime = System.currentTimeMillis();
        return asyncLoader.apply(key).thenApply(value -> {
            long loadTime = System.currentTimeMillis() - startTime;
            this.loadCount.increment();
            this.totalLoadTime.add(loadTime);
            if (value != null && !value.isEmpty()) {
                this.put(key, (List<Material>)value);
            }
            return value != null ? new ArrayList(value) : new ArrayList();
        });
    }

    public void put(String key, List<Material> value) {
        Objects.requireNonNull(key, "Key cannot be null");
        Objects.requireNonNull(value, "Value cannot be null");
        this.ensureNotClosed();
        if (this.cache.size() >= this.maxSize && !this.cache.containsKey(key)) {
            this.performSizeBasedEviction();
        }
        this.cache.put(key, CacheEntry.of(value));
    }

    public boolean invalidate(String key) {
        this.ensureNotClosed();
        CacheEntry removed = this.cache.remove(key);
        if (removed != null) {
            this.evictionCount.increment();
        }
        return removed != null;
    }

    public void invalidateAll() {
        this.ensureNotClosed();
        int size = this.cache.size();
        this.cache.clear();
        this.evictionCount.add(size);
        LOGGER.info("Cache cleared: " + size + " entries invalidated");
    }

    public int invalidateIf(Predicate<String> predicate) {
        this.ensureNotClosed();
        int removed = 0;
        Iterator<String> iterator = this.cache.keySet().iterator();
        while (iterator.hasNext()) {
            if (!predicate.test(iterator.next())) continue;
            iterator.remove();
            this.evictionCount.increment();
            ++removed;
        }
        return removed;
    }

    public CompletableFuture<List<Material>> refresh(String key, Function<String, List<Material>> loader) {
        return CompletableFuture.supplyAsync(() -> {
            this.invalidate(key);
            return this.get(key, loader);
        }, this.loadingExecutor);
    }

    public CompletableFuture<Void> warmUp(Collection<String> keys, Function<String, List<Material>> loader) {
        List<CompletableFuture> futures = keys.stream().map(key -> CompletableFuture.runAsync(() -> this.get((String)key, loader), this.loadingExecutor)).toList();
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
    }

    public int size() {
        return this.cache.size();
    }

    public boolean isEmpty() {
        return this.cache.isEmpty();
    }

    public boolean containsKey(String key) {
        CacheEntry entry = this.cache.get(key);
        return entry != null && !entry.isExpired(this.ttl);
    }

    public CacheStats getStats() {
        long misses;
        long hits = this.hitCount.sum();
        long total = hits + (misses = this.missCount.sum());
        double hitRate = total > 0L ? (double)hits / (double)total : 0.0;
        double avgLoadTime = this.loadCount.sum() > 0L ? (double)this.totalLoadTime.sum() / (double)this.loadCount.sum() : 0.0;
        return new CacheStats(hits, misses, this.evictionCount.sum(), this.loadCount.sum(), hitRate, avgLoadTime, this.cache.size(), total);
    }

    public void resetStats() {
        this.hitCount.reset();
        this.missCount.reset();
        this.evictionCount.reset();
        this.loadCount.reset();
        this.totalLoadTime.reset();
        LOGGER.info("Cache statistics reset");
    }

    private void ensureNotClosed() {
        if (this.closed) {
            throw new IllegalStateException("Cache has been closed");
        }
    }

    @Override
    public void close() {
        if (!this.closed) {
            this.closed = true;
            LOGGER.info("Closing cache. Final stats: " + this.getStats().getSummary());
            this.cache.clear();
            this.loadingExecutor.shutdown();
            this.maintenanceExecutor.shutdown();
            try {
                if (!this.loadingExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.loadingExecutor.shutdownNow();
                }
                if (!this.maintenanceExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.maintenanceExecutor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.loadingExecutor.shutdownNow();
                this.maintenanceExecutor.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
    }

    private record CacheEntry(List<Material> value, Instant createdAt, Instant lastAccessedAt, AtomicLong accessCount) {
        public static CacheEntry of(List<Material> value) {
            return new CacheEntry(Collections.unmodifiableList(new ArrayList<Material>(value)), Instant.now(), Instant.now(), new AtomicLong(1L));
        }

        public CacheEntry recordAccess() {
            this.accessCount.incrementAndGet();
            return new CacheEntry(this.value, this.createdAt, Instant.now(), this.accessCount);
        }

        public boolean isExpired(Duration ttl) {
            return Duration.between(this.createdAt, Instant.now()).compareTo(ttl) > 0;
        }

        public boolean isStale(Duration idleTime) {
            return Duration.between(this.lastAccessedAt, Instant.now()).compareTo(idleTime) > 0;
        }
    }

    public record CacheStats(long hitCount, long missCount, long evictionCount, long loadCount, double hitRate, double averageLoadTime, int size, long totalAccessCount) {
        public String getSummary() {
            return String.format("Cache Statistics:\n- Hit Rate: %.2f%%\n- Total Hits: %d\n- Total Misses: %d\n- Evictions: %d\n- Cache Size: %d\n- Average Load Time: %.2f ms\n- Total Accesses: %d\n", this.hitRate * 100.0, this.hitCount, this.missCount, this.evictionCount, this.size, this.averageLoadTime, this.totalAccessCount);
        }
    }
}

