/*
 * Decompiled with CFR 0.152.
 */
package com.university.bookstore.repository;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.university.bookstore.model.Material;
import com.university.bookstore.repository.MaterialRepository;
import com.university.bookstore.repository.MaterialsWrapper;
import com.university.bookstore.repository.RepositoryException;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;
import java.util.logging.Logger;

public class ModernJsonMaterialRepository
implements MaterialRepository,
AutoCloseable {
    private static final Logger LOGGER = Logger.getLogger(ModernJsonMaterialRepository.class.getName());
    private static final String SAFE_BASE_DIR = System.getProperty("user.dir") + "/data";
    private static final int MAX_PATH_LENGTH = 255;
    private final Path dataFile;
    private final Path backupFile;
    private final ObjectMapper objectMapper;
    private final ReadWriteLock fileLock;
    private volatile boolean closed = false;

    public ModernJsonMaterialRepository(String filePath) {
        Path validatedPath;
        this.dataFile = validatedPath = this.validateAndSanitizePath(filePath);
        this.backupFile = Paths.get(validatedPath.toString() + ".backup", new String[0]);
        this.fileLock = new ReentrantReadWriteLock();
        this.objectMapper = new ObjectMapper();
        this.objectMapper.enable(SerializationFeature.INDENT_OUTPUT);
        this.objectMapper.findAndRegisterModules();
        this.initializeStorage();
    }

    private void initializeStorage() {
        try {
            Path parentDir = this.dataFile.getParent();
            if (parentDir != null && !Files.exists(parentDir, new LinkOption[0])) {
                Files.createDirectories(parentDir, new FileAttribute[0]);
                LOGGER.info("Created directory: " + String.valueOf(parentDir));
            }
            if (!Files.exists(this.dataFile, new LinkOption[0])) {
                this.saveAtomic(new ArrayList<Material>());
                LOGGER.info("Created new data file: " + String.valueOf(this.dataFile));
            }
        }
        catch (IOException e) {
            throw new RepositoryException("Failed to initialize storage", e);
        }
    }

    private Path validateAndSanitizePath(String filePath) {
        if (filePath == null || filePath.trim().isEmpty()) {
            throw new IllegalArgumentException("File path cannot be null or empty");
        }
        String cleanPath = filePath.replaceAll("\\.\\./", "").replaceAll("\\.\\.", "");
        if (cleanPath.contains("../") || cleanPath.contains("..\\") || cleanPath.contains("%2e%2e") || cleanPath.contains("%252e")) {
            throw new SecurityException("Invalid file path: potential path traversal detected");
        }
        if (cleanPath.length() > 255) {
            throw new IllegalArgumentException("File path exceeds maximum length");
        }
        try {
            Path normalizedPath = Paths.get(cleanPath, new String[0]).normalize();
            Path safePath = Paths.get(SAFE_BASE_DIR, new String[0]).normalize();
            if (!normalizedPath.isAbsolute()) {
                normalizedPath = safePath.resolve(normalizedPath).normalize();
            }
            if (!normalizedPath.startsWith(safePath)) {
                throw new SecurityException("File path must be within the safe directory: " + SAFE_BASE_DIR);
            }
            return normalizedPath;
        }
        catch (Exception e) {
            if (e instanceof SecurityException) {
                throw (SecurityException)e;
            }
            throw new IllegalArgumentException("Invalid file path: " + e.getMessage(), e);
        }
    }

    @Override
    public void save(Material material) {
        if (material == null) {
            throw new IllegalArgumentException("Material cannot be null");
        }
        this.ensureNotClosed();
        this.fileLock.writeLock().lock();
        try {
            List<Material> materials = this.loadAllInternal();
            materials.removeIf(m -> m.getId().equals(material.getId()));
            materials.add(material);
            this.saveAtomic(materials);
            LOGGER.fine("Saved material: " + material.getId());
        }
        catch (IOException e) {
            throw new RepositoryException("Failed to save material: " + material.getId(), e);
        }
        finally {
            this.fileLock.writeLock().unlock();
        }
    }

    public void saveAll(Collection<Material> materialsToSave) {
        Objects.requireNonNull(materialsToSave, "Materials collection cannot be null");
        this.ensureNotClosed();
        if (materialsToSave.isEmpty()) {
            return;
        }
        this.fileLock.writeLock().lock();
        try {
            List<Material> materials = this.loadAllInternal();
            HashMap<String, Material> materialMap = new HashMap<String, Material>();
            for (Material m : materials) {
                materialMap.put(m.getId(), m);
            }
            for (Material material : materialsToSave) {
                if (material == null) continue;
                materialMap.put(material.getId(), material);
            }
            this.saveAtomic(new ArrayList<Material>(materialMap.values()));
            LOGGER.fine("Saved " + materialsToSave.size() + " materials in batch");
        }
        catch (IOException e) {
            throw new RepositoryException("Failed to save materials batch", e);
        }
        finally {
            this.fileLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Optional<Material> findById(String id) {
        if (id == null || id.trim().isEmpty()) {
            return Optional.empty();
        }
        this.ensureNotClosed();
        this.fileLock.readLock().lock();
        try {
            List<Material> materials = this.loadAllInternal();
            Optional<Material> optional = materials.stream().filter(m -> id.equals(m.getId())).findFirst();
            return optional;
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "Failed to find material by ID: " + id, e);
            Optional<Material> optional = Optional.empty();
            return optional;
        }
        finally {
            this.fileLock.readLock().unlock();
        }
    }

    @Override
    public List<Material> findAll() {
        this.ensureNotClosed();
        this.fileLock.readLock().lock();
        try {
            ArrayList<Material> arrayList = new ArrayList<Material>(this.loadAllInternal());
            return arrayList;
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "Failed to load all materials", e);
            ArrayList<Material> arrayList = new ArrayList<Material>();
            return arrayList;
        }
        finally {
            this.fileLock.readLock().unlock();
        }
    }

    @Override
    public boolean delete(String id) {
        if (id == null || id.trim().isEmpty()) {
            return false;
        }
        this.ensureNotClosed();
        this.fileLock.writeLock().lock();
        try {
            List<Material> materials = this.loadAllInternal();
            boolean removed = materials.removeIf(m -> id.equals(m.getId()));
            if (removed) {
                this.saveAtomic(materials);
                LOGGER.fine("Deleted material: " + id);
            }
            boolean bl = removed;
            return bl;
        }
        catch (IOException e) {
            throw new RepositoryException("Failed to delete material: " + id, e);
        }
        finally {
            this.fileLock.writeLock().unlock();
        }
    }

    public int deleteAll(Collection<String> ids) {
        if (ids == null || ids.isEmpty()) {
            return 0;
        }
        this.ensureNotClosed();
        this.fileLock.writeLock().lock();
        try {
            List<Material> materials = this.loadAllInternal();
            HashSet<String> idSet = new HashSet<String>(ids);
            int originalSize = materials.size();
            materials.removeIf(m -> idSet.contains(m.getId()));
            int deleted = originalSize - materials.size();
            if (deleted > 0) {
                this.saveAtomic(materials);
                LOGGER.fine("Deleted " + deleted + " materials in batch");
            }
            int n = deleted;
            return n;
        }
        catch (IOException e) {
            throw new RepositoryException("Failed to delete materials batch", e);
        }
        finally {
            this.fileLock.writeLock().unlock();
        }
    }

    @Override
    public boolean exists(String id) {
        return this.findById(id).isPresent();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long count() {
        this.ensureNotClosed();
        this.fileLock.readLock().lock();
        try {
            long l = this.loadAllInternal().size();
            return l;
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "Failed to count materials", e);
            long l = 0L;
            return l;
        }
        finally {
            this.fileLock.readLock().unlock();
        }
    }

    @Override
    public void deleteAll() {
        this.ensureNotClosed();
        this.fileLock.writeLock().lock();
        try {
            this.saveAtomic(new ArrayList<Material>());
            LOGGER.info("Cleared all materials from repository");
        }
        catch (IOException e) {
            throw new RepositoryException("Failed to clear all materials", e);
        }
        finally {
            this.fileLock.writeLock().unlock();
        }
    }

    private List<Material> loadAllInternal() throws IOException {
        ArrayList<Material> arrayList;
        block18: {
            if (!Files.exists(this.dataFile, new LinkOption[0]) || Files.size(this.dataFile) == 0L) {
                return new ArrayList<Material>();
            }
            BufferedReader reader = Files.newBufferedReader(this.dataFile);
            try {
                MaterialsWrapper wrapper = (MaterialsWrapper)this.objectMapper.readValue((Reader)reader, MaterialsWrapper.class);
                ArrayList<Material> arrayList2 = arrayList = wrapper.getMaterials() != null ? wrapper.getMaterials() : new ArrayList<Material>();
                if (reader == null) break block18;
            }
            catch (Throwable wrapper) {
                try {
                    if (reader != null) {
                        try {
                            reader.close();
                        }
                        catch (Throwable throwable) {
                            wrapper.addSuppressed(throwable);
                        }
                    }
                    throw wrapper;
                }
                catch (IOException e) {
                    if (Files.exists(this.backupFile, new LinkOption[0])) {
                        ArrayList<Material> arrayList3;
                        block19: {
                            LOGGER.warning("Main file corrupted, attempting to restore from backup");
                            BufferedReader backupReader = Files.newBufferedReader(this.backupFile);
                            try {
                                MaterialsWrapper wrapper2 = (MaterialsWrapper)this.objectMapper.readValue((Reader)backupReader, MaterialsWrapper.class);
                                Files.copy(this.backupFile, this.dataFile, StandardCopyOption.REPLACE_EXISTING);
                                ArrayList<Material> arrayList4 = arrayList3 = wrapper2.getMaterials() != null ? wrapper2.getMaterials() : new ArrayList<Material>();
                                if (backupReader == null) break block19;
                            }
                            catch (Throwable throwable) {
                                try {
                                    if (backupReader != null) {
                                        try {
                                            backupReader.close();
                                        }
                                        catch (Throwable throwable2) {
                                            throwable.addSuppressed(throwable2);
                                        }
                                    }
                                    throw throwable;
                                }
                                catch (IOException backupError) {
                                    LOGGER.log(Level.SEVERE, "Failed to restore from backup", backupError);
                                    return new ArrayList<Material>();
                                }
                            }
                            backupReader.close();
                        }
                        return arrayList3;
                    }
                    LOGGER.log(Level.WARNING, "Failed to parse JSON, returning empty list", e);
                    return new ArrayList<Material>();
                }
            }
            reader.close();
        }
        return arrayList;
    }

    private void saveAtomic(List<Material> materials) throws IOException {
        if (Files.exists(this.dataFile, new LinkOption[0])) {
            Files.copy(this.dataFile, this.backupFile, StandardCopyOption.REPLACE_EXISTING);
        }
        Path tempFile = Files.createTempFile(this.dataFile.getParent(), "temp", ".json", new FileAttribute[0]);
        try {
            try (BufferedWriter writer = Files.newBufferedWriter(tempFile, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);){
                MaterialsWrapper wrapper = new MaterialsWrapper(materials);
                this.objectMapper.writerWithDefaultPrettyPrinter().writeValue((Writer)writer, (Object)wrapper);
            }
            Files.move(tempFile, this.dataFile, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (IOException e) {
            try {
                Files.deleteIfExists(tempFile);
            }
            catch (IOException deleteError) {
                LOGGER.log(Level.WARNING, "Failed to delete temp file", deleteError);
            }
            throw e;
        }
    }

    public boolean createBackup() {
        this.ensureNotClosed();
        this.fileLock.readLock().lock();
        try {
            if (Files.exists(this.dataFile, new LinkOption[0])) {
                Path backupPath = Paths.get(String.valueOf(this.dataFile) + "." + System.currentTimeMillis() + ".backup", new String[0]);
                Files.copy(this.dataFile, backupPath, new CopyOption[0]);
                LOGGER.info("Created backup: " + String.valueOf(backupPath));
                boolean bl = true;
                return bl;
            }
            boolean backupPath = false;
            return backupPath;
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "Failed to create backup", e);
            boolean bl = false;
            return bl;
        }
        finally {
            this.fileLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean restoreFromBackup(String backupPath) {
        Objects.requireNonNull(backupPath, "Backup path cannot be null");
        this.ensureNotClosed();
        Path backup = Paths.get(backupPath, new String[0]);
        if (!Files.exists(backup, new LinkOption[0])) {
            LOGGER.warning("Backup file does not exist: " + backupPath);
            return false;
        }
        this.fileLock.writeLock().lock();
        try {
            try (BufferedReader reader = Files.newBufferedReader(backup);){
                MaterialsWrapper wrapper = (MaterialsWrapper)this.objectMapper.readValue((Reader)reader, MaterialsWrapper.class);
            }
            Files.copy(backup, this.dataFile, StandardCopyOption.REPLACE_EXISTING);
            LOGGER.info("Restored from backup: " + backupPath);
            boolean bl = true;
            return bl;
        }
        catch (IOException e) {
            LOGGER.log(Level.SEVERE, "Failed to restore from backup: " + e.getMessage(), e);
            e.printStackTrace();
            boolean bl = false;
            return bl;
        }
        finally {
            this.fileLock.writeLock().unlock();
        }
    }

    public String getFilePath() {
        return this.dataFile.toString();
    }

    public boolean dataFileExists() {
        return Files.exists(this.dataFile, new LinkOption[0]);
    }

    public long getDataFileSize() {
        try {
            return Files.size(this.dataFile);
        }
        catch (IOException e) {
            return 0L;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void performMaintenance() {
        this.ensureNotClosed();
        this.fileLock.writeLock().lock();
        try {
            Path parent = this.dataFile.getParent();
            if (parent != null) {
                long cutoffTime = System.currentTimeMillis() - 604800000L;
                Files.list(parent).filter(path -> path.toString().endsWith(".backup")).filter(path -> {
                    try {
                        return Files.getLastModifiedTime(path, new LinkOption[0]).toMillis() < cutoffTime;
                    }
                    catch (IOException e) {
                        return false;
                    }
                }).forEach(path -> {
                    try {
                        Files.delete(path);
                        LOGGER.fine("Deleted old backup: " + String.valueOf(path));
                    }
                    catch (IOException e) {
                        LOGGER.warning("Failed to delete old backup: " + String.valueOf(path));
                    }
                });
            }
            List<Material> materials = this.loadAllInternal();
            this.saveAtomic(materials);
            LOGGER.info("Maintenance completed");
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "Maintenance failed", e);
        }
        finally {
            this.fileLock.writeLock().unlock();
        }
    }

    private void ensureNotClosed() {
        if (this.closed) {
            throw new IllegalStateException("Repository has been closed");
        }
    }

    @Override
    public void close() {
        if (!this.closed) {
            this.closed = true;
            LOGGER.info("Repository closed");
        }
    }
}

