/*
 * Decompiled with CFR 0.152.
 */
package com.university.bookstore.repository;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.university.bookstore.model.Material;
import com.university.bookstore.repository.MaterialRepository;
import com.university.bookstore.repository.MaterialsWrapper;
import com.university.bookstore.repository.RepositoryException;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

public class JsonMaterialRepository
implements MaterialRepository {
    private static final String SAFE_BASE_DIR = System.getProperty("user.dir") + "/data";
    private static final int MAX_PATH_LENGTH = 255;
    private final String filePath;
    private final ObjectMapper objectMapper;
    private final File dataFile;

    public JsonMaterialRepository(String filePath) {
        this.filePath = this.validateAndSanitizePath(filePath);
        this.objectMapper = new ObjectMapper();
        this.objectMapper.findAndRegisterModules();
        this.dataFile = new File(this.filePath);
        File parentDir = this.dataFile.getParentFile();
        if (parentDir != null && !parentDir.exists()) {
            parentDir.mkdirs();
        }
    }

    private String validateAndSanitizePath(String filePath) {
        if (filePath == null || filePath.trim().isEmpty()) {
            throw new IllegalArgumentException("File path cannot be null or empty");
        }
        if (filePath.contains("../") || filePath.contains("..\\") || filePath.contains("%2e%2e") || filePath.contains("%252e")) {
            throw new SecurityException("Invalid file path: potential path traversal detected");
        }
        String cleanPath = filePath.replaceAll("\\.\\./", "").replaceAll("\\.\\.", "");
        if (cleanPath.length() > 255) {
            throw new IllegalArgumentException("File path exceeds maximum length");
        }
        try {
            Path normalizedPath = Paths.get(cleanPath, new String[0]).normalize();
            Path safePath = Paths.get(SAFE_BASE_DIR, new String[0]).normalize();
            if (!normalizedPath.isAbsolute()) {
                normalizedPath = safePath.resolve(normalizedPath).normalize();
            }
            if (!normalizedPath.startsWith(safePath)) {
                throw new SecurityException("File path must be within the safe directory: " + SAFE_BASE_DIR);
            }
            return normalizedPath.toString();
        }
        catch (Exception e) {
            if (e instanceof SecurityException) {
                throw (SecurityException)e;
            }
            throw new IllegalArgumentException("Invalid file path: " + e.getMessage(), e);
        }
    }

    @Override
    public void save(Material material) {
        if (material == null) {
            throw new IllegalArgumentException("Material cannot be null");
        }
        try {
            List<Material> materials = this.loadAll();
            materials.removeIf(m -> m.getId().equals(material.getId()));
            materials.add(material);
            MaterialsWrapper wrapper = new MaterialsWrapper(materials);
            this.objectMapper.writerWithDefaultPrettyPrinter().writeValue(this.dataFile, (Object)wrapper);
        }
        catch (IOException e) {
            throw new RepositoryException("Failed to save material: " + material.getId(), e);
        }
    }

    @Override
    public Optional<Material> findById(String id) {
        if (id == null || id.trim().isEmpty()) {
            return Optional.empty();
        }
        try {
            List<Material> materials = this.loadAll();
            return materials.stream().filter(m -> id.equals(m.getId())).findFirst();
        }
        catch (IOException e) {
            throw new RepositoryException("Failed to find material by ID: " + id, e);
        }
    }

    @Override
    public List<Material> findAll() {
        try {
            return this.loadAll();
        }
        catch (IOException e) {
            throw new RepositoryException("Failed to load all materials", e);
        }
    }

    @Override
    public boolean delete(String id) {
        if (id == null || id.trim().isEmpty()) {
            return false;
        }
        try {
            List<Material> materials = this.loadAll();
            boolean removed = materials.removeIf(m -> id.equals(m.getId()));
            if (removed) {
                MaterialsWrapper wrapper = new MaterialsWrapper(materials);
                this.objectMapper.writerWithDefaultPrettyPrinter().writeValue(this.dataFile, (Object)wrapper);
            }
            return removed;
        }
        catch (IOException e) {
            throw new RepositoryException("Failed to delete material: " + id, e);
        }
    }

    @Override
    public boolean exists(String id) {
        return this.findById(id).isPresent();
    }

    @Override
    public long count() {
        try {
            return this.loadAll().size();
        }
        catch (IOException e) {
            throw new RepositoryException("Failed to count materials", e);
        }
    }

    @Override
    public void deleteAll() {
        try {
            MaterialsWrapper wrapper = new MaterialsWrapper(new ArrayList<Material>());
            this.objectMapper.writerWithDefaultPrettyPrinter().writeValue(this.dataFile, (Object)wrapper);
        }
        catch (IOException e) {
            throw new RepositoryException("Failed to clear all materials", e);
        }
    }

    private List<Material> loadAll() throws IOException {
        if (!this.dataFile.exists()) {
            return new ArrayList<Material>();
        }
        if (this.dataFile.length() == 0L) {
            return new ArrayList<Material>();
        }
        try {
            MaterialsWrapper wrapper = (MaterialsWrapper)this.objectMapper.readValue(this.dataFile, MaterialsWrapper.class);
            return wrapper.getMaterials() != null ? wrapper.getMaterials() : new ArrayList<Material>();
        }
        catch (IOException e) {
            System.err.println("Failed to load materials from " + String.valueOf(this.dataFile) + ": " + e.getMessage());
            return new ArrayList<Material>();
        }
    }

    public String getFilePath() {
        return this.filePath;
    }

    public boolean dataFileExists() {
        return this.dataFile.exists();
    }

    public long getDataFileSize() {
        return this.dataFile.length();
    }
}

