/*
 * Decompiled with CFR 0.152.
 */
package com.university.bookstore.impl;

import com.university.bookstore.api.MaterialStore;
import com.university.bookstore.model.Magazine;
import com.university.bookstore.model.Material;
import com.university.bookstore.model.Media;
import com.university.bookstore.model.PrintedBook;
import java.time.Year;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.StampedLock;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ModernConcurrentMaterialStore
implements MaterialStore,
AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(ModernConcurrentMaterialStore.class);
    private final Map<String, Material> materials = new ConcurrentHashMap<String, Material>();
    private final StampedLock stampedLock = new StampedLock();
    private final ExecutorService executorService = new ForkJoinPool(Runtime.getRuntime().availableProcessors(), ForkJoinPool.defaultForkJoinWorkerThreadFactory, null, true);
    private final ScheduledExecutorService scheduledExecutor = Executors.newScheduledThreadPool(2, r -> {
        Thread t = new Thread(r, "MaterialStore-Scheduler");
        t.setDaemon(true);
        return t;
    });
    private volatile boolean closed = false;

    public ModernConcurrentMaterialStore() {
        this.scheduleMaintenanceTasks();
    }

    public ModernConcurrentMaterialStore(Collection<Material> initialMaterials) {
        this();
        if (initialMaterials != null) {
            initialMaterials.parallelStream().forEach(this::addMaterial);
        }
    }

    private void scheduleMaintenanceTasks() {
        this.scheduledExecutor.scheduleAtFixedRate(this::performMaintenance, 1L, 1L, TimeUnit.HOURS);
    }

    private void performMaintenance() {
        if (!this.closed) {
            // empty if block
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean addMaterial(Material material) {
        Objects.requireNonNull(material, "Material cannot be null");
        this.ensureNotClosed();
        long stamp = this.stampedLock.writeLock();
        try {
            boolean bl = this.materials.putIfAbsent(material.getId(), material) == null;
            return bl;
        }
        finally {
            this.stampedLock.unlockWrite(stamp);
        }
    }

    public CompletableFuture<Boolean> addMaterialAsync(Material material) {
        return CompletableFuture.supplyAsync(() -> this.addMaterial(material), this.executorService);
    }

    public CompletableFuture<Map<String, Boolean>> addMaterialsBatchAsync(Collection<Material> materials) {
        List<CompletableFuture> futures = materials.stream().map(material -> CompletableFuture.supplyAsync(() -> Map.entry(material.getId(), this.addMaterial((Material)material)), this.executorService)).toList();
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).thenApply(v -> futures.stream().map(CompletableFuture::join).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue)));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Optional<Material> removeMaterial(String id) {
        if (id == null) {
            return Optional.empty();
        }
        this.ensureNotClosed();
        long stamp = this.stampedLock.writeLock();
        try {
            Optional<Material> optional = Optional.ofNullable(this.materials.remove(id));
            return optional;
        }
        finally {
            this.stampedLock.unlockWrite(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Optional<Material> findById(String id) {
        if (id == null) {
            return Optional.empty();
        }
        this.ensureNotClosed();
        long stamp = this.stampedLock.tryOptimisticRead();
        Material material = this.materials.get(id);
        if (!this.stampedLock.validate(stamp)) {
            stamp = this.stampedLock.readLock();
            try {
                material = this.materials.get(id);
            }
            finally {
                this.stampedLock.unlockRead(stamp);
            }
        }
        return Optional.ofNullable(material);
    }

    public CompletableFuture<Optional<Material>> findByIdAsync(String id) {
        return CompletableFuture.supplyAsync(() -> this.findById(id), this.executorService);
    }

    public CompletableFuture<Map<String, Material>> findByIdsAsync(List<String> ids) {
        List<CompletableFuture> futures = ids.stream().map(id -> CompletableFuture.supplyAsync(() -> Map.entry(id, this.findById((String)id)), this.executorService)).toList();
        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0])).thenApply(v -> futures.stream().map(CompletableFuture::join).filter(entry -> ((Optional)entry.getValue()).isPresent()).collect(Collectors.toMap(Map.Entry::getKey, entry -> (Material)((Optional)entry.getValue()).get())));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<Material> searchByTitle(String title) {
        if (title == null || title.trim().isEmpty()) {
            return new ArrayList<Material>();
        }
        this.ensureNotClosed();
        String searchTerm = title.toLowerCase().trim();
        long stamp = this.stampedLock.readLock();
        try {
            List<Material> list = this.materials.values().parallelStream().filter(m -> m.getTitle().toLowerCase().contains(searchTerm)).collect(Collectors.toList());
            return list;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    public CompletableFuture<List<Material>> searchByTitleAsync(String title) {
        return CompletableFuture.supplyAsync(() -> this.searchByTitle(title), this.executorService);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<Material> searchByCreator(String creator) {
        if (creator == null || creator.trim().isEmpty()) {
            return new ArrayList<Material>();
        }
        this.ensureNotClosed();
        String searchTerm = creator.toLowerCase().trim();
        long stamp = this.stampedLock.readLock();
        try {
            List<Material> list = this.materials.values().parallelStream().filter(m -> m.getCreator().toLowerCase().contains(searchTerm)).collect(Collectors.toList());
            return list;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<Material> getMaterialsByType(Material.MaterialType type) {
        if (type == null) {
            return new ArrayList<Material>();
        }
        this.ensureNotClosed();
        long stamp = this.stampedLock.readLock();
        try {
            List<Material> list = this.materials.values().parallelStream().filter(m -> m.getType() == type).collect(Collectors.toList());
            return list;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<Media> getMediaMaterials() {
        this.ensureNotClosed();
        long stamp = this.stampedLock.readLock();
        try {
            List<Media> list = this.materials.values().parallelStream().filter(Media.class::isInstance).map(Media.class::cast).collect(Collectors.toList());
            return list;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<Material> filterMaterials(Predicate<Material> predicate) {
        Objects.requireNonNull(predicate, "Predicate cannot be null");
        this.ensureNotClosed();
        long stamp = this.stampedLock.readLock();
        try {
            List<Material> list = this.materials.values().parallelStream().filter(predicate).collect(Collectors.toList());
            return list;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<Material> getMaterialsByPriceRange(double minPrice, double maxPrice) {
        if (minPrice < 0.0 || maxPrice < 0.0 || minPrice > maxPrice) {
            return new ArrayList<Material>();
        }
        this.ensureNotClosed();
        long stamp = this.stampedLock.readLock();
        try {
            List<Material> list = this.materials.values().parallelStream().filter(m -> m.getPrice() >= minPrice && m.getPrice() <= maxPrice).collect(Collectors.toList());
            return list;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<Material> getMaterialsByYear(int year) {
        this.ensureNotClosed();
        long stamp = this.stampedLock.readLock();
        try {
            List<Material> list = this.materials.values().parallelStream().filter(m -> m.getYear() == year).collect(Collectors.toList());
            return list;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<Material> getAllMaterialsSorted() {
        this.ensureNotClosed();
        long stamp = this.stampedLock.readLock();
        try {
            List<Material> list = this.materials.values().stream().sorted().collect(Collectors.toList());
            return list;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<Material> getAllMaterials() {
        this.ensureNotClosed();
        long stamp = this.stampedLock.tryOptimisticRead();
        ArrayList<Material> result = new ArrayList<Material>(this.materials.values());
        if (!this.stampedLock.validate(stamp)) {
            stamp = this.stampedLock.readLock();
            try {
                result = new ArrayList<Material>(this.materials.values());
            }
            finally {
                this.stampedLock.unlockRead(stamp);
            }
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public double getTotalInventoryValue() {
        this.ensureNotClosed();
        long stamp = this.stampedLock.readLock();
        try {
            double d = this.materials.values().parallelStream().mapToDouble(Material::getPrice).sum();
            return d;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    public CompletableFuture<Double> getTotalInventoryValueAsync() {
        return CompletableFuture.supplyAsync(this::getTotalInventoryValue, this.executorService);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public double getTotalDiscountedValue() {
        this.ensureNotClosed();
        long stamp = this.stampedLock.readLock();
        try {
            double d = this.materials.values().parallelStream().mapToDouble(Material::getDiscountedPrice).sum();
            return d;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public MaterialStore.InventoryStats getInventoryStats() {
        this.ensureNotClosed();
        long stamp = this.stampedLock.readLock();
        try {
            if (this.materials.isEmpty()) {
                MaterialStore.InventoryStats inventoryStats = new MaterialStore.InventoryStats(0, 0.0, 0.0, 0, 0, 0);
                return inventoryStats;
            }
            List prices = this.materials.values().stream().map(Material::getPrice).sorted().collect(Collectors.toList());
            double averagePrice = prices.stream().mapToDouble(Double::doubleValue).average().orElse(0.0);
            double medianPrice = prices.size() % 2 == 0 ? ((Double)prices.get(prices.size() / 2 - 1) + (Double)prices.get(prices.size() / 2)) / 2.0 : (Double)prices.get(prices.size() / 2);
            int uniqueTypes = (int)this.materials.values().stream().map(Material::getType).distinct().count();
            int mediaCount = (int)this.materials.values().stream().filter(Media.class::isInstance).count();
            int printCount = (int)this.materials.values().stream().filter(m -> m instanceof PrintedBook || m instanceof Magazine).count();
            MaterialStore.InventoryStats inventoryStats = new MaterialStore.InventoryStats(this.materials.size(), averagePrice, medianPrice, uniqueTypes, mediaCount, printCount);
            return inventoryStats;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    public CompletableFuture<MaterialStore.InventoryStats> getInventoryStatsAsync() {
        return CompletableFuture.supplyAsync(this::getInventoryStats, this.executorService);
    }

    @Override
    public void clearInventory() {
        this.ensureNotClosed();
        long stamp = this.stampedLock.writeLock();
        try {
            this.materials.clear();
        }
        finally {
            this.stampedLock.unlockWrite(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public int size() {
        this.ensureNotClosed();
        long stamp = this.stampedLock.tryOptimisticRead();
        int size = this.materials.size();
        if (!this.stampedLock.validate(stamp)) {
            stamp = this.stampedLock.readLock();
            try {
                size = this.materials.size();
            }
            finally {
                this.stampedLock.unlockRead(stamp);
            }
        }
        return size;
    }

    @Override
    public boolean isEmpty() {
        return this.size() == 0;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<Material> findRecentMaterials(int years) {
        if (years < 0) {
            throw new IllegalArgumentException("Years cannot be negative: " + years);
        }
        this.ensureNotClosed();
        int currentYear = Year.now().getValue();
        int cutoffYear = currentYear - years;
        long stamp = this.stampedLock.readLock();
        try {
            List<Material> list = this.materials.values().parallelStream().filter(material -> material.getYear() >= cutoffYear).collect(Collectors.toList());
            return list;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<Material> findByCreators(String ... creators) {
        if (creators == null || creators.length == 0) {
            return new ArrayList<Material>();
        }
        this.ensureNotClosed();
        Set creatorSet = Arrays.stream(creators).filter(Objects::nonNull).map(String::trim).filter(s -> !s.isEmpty()).collect(Collectors.toSet());
        if (creatorSet.isEmpty()) {
            return new ArrayList<Material>();
        }
        long stamp = this.stampedLock.readLock();
        try {
            List<Material> list = this.materials.values().parallelStream().filter(material -> creatorSet.contains(material.getCreator())).collect(Collectors.toList());
            return list;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<Material> findWithPredicate(Predicate<Material> condition) {
        Objects.requireNonNull(condition, "Predicate cannot be null");
        this.ensureNotClosed();
        long stamp = this.stampedLock.readLock();
        try {
            List<Material> list = this.materials.values().parallelStream().filter(condition).collect(Collectors.toList());
            return list;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<Material> getSorted(Comparator<Material> comparator) {
        Objects.requireNonNull(comparator, "Comparator cannot be null");
        this.ensureNotClosed();
        long stamp = this.stampedLock.readLock();
        try {
            List<Material> list = this.materials.values().stream().sorted(comparator).collect(Collectors.toList());
            return list;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    public CompletableFuture<List<Material>> parallelSearchAsync(String title, String creator, Material.MaterialType type) {
        ArrayList<CompletableFuture<List<Material>>> searches = new ArrayList<CompletableFuture<List<Material>>>();
        if (title != null && !title.trim().isEmpty()) {
            searches.add(this.searchByTitleAsync(title));
        }
        if (creator != null && !creator.trim().isEmpty()) {
            searches.add(CompletableFuture.supplyAsync(() -> this.searchByCreator(creator), this.executorService));
        }
        if (type != null) {
            searches.add(CompletableFuture.supplyAsync(() -> this.getMaterialsByType(type), this.executorService));
        }
        if (searches.isEmpty()) {
            return CompletableFuture.completedFuture(new ArrayList());
        }
        return CompletableFuture.allOf(searches.toArray(new CompletableFuture[0])).thenApply(v -> searches.stream().map(CompletableFuture::join).flatMap(Collection::stream).distinct().collect(Collectors.toList()));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Map<Material.MaterialType, List<Material>> groupByType() {
        this.ensureNotClosed();
        long stamp = this.stampedLock.readLock();
        try {
            Map<Material.MaterialType, List<Material>> map = this.materials.values().stream().collect(Collectors.groupingBy(Material::getType));
            return map;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<Material> getDiscountedMaterials() {
        this.ensureNotClosed();
        long stamp = this.stampedLock.readLock();
        try {
            List<Material> list = this.materials.values().parallelStream().filter(m -> m.getDiscountRate() > 0.0).collect(Collectors.toList());
            return list;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public double getTotalDiscountAmount() {
        this.ensureNotClosed();
        long stamp = this.stampedLock.readLock();
        try {
            double d = this.materials.values().parallelStream().mapToDouble(m -> m.getPrice() * m.getDiscountRate()).sum();
            return d;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }

    private void ensureNotClosed() {
        if (this.closed) {
            throw new IllegalStateException("MaterialStore has been closed");
        }
    }

    @Override
    public void close() {
        if (!this.closed) {
            this.closed = true;
            this.executorService.shutdown();
            this.scheduledExecutor.shutdown();
            try {
                if (!this.executorService.awaitTermination(10L, TimeUnit.SECONDS)) {
                    this.executorService.shutdownNow();
                    if (!this.executorService.awaitTermination(5L, TimeUnit.SECONDS)) {
                        LOGGER.error("ExecutorService did not terminate within timeout");
                    }
                }
                if (!this.scheduledExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    this.scheduledExecutor.shutdownNow();
                    if (!this.scheduledExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                        LOGGER.error("ScheduledExecutor did not terminate within timeout");
                    }
                }
            }
            catch (InterruptedException e) {
                this.executorService.shutdownNow();
                this.scheduledExecutor.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String toString() {
        this.ensureNotClosed();
        long stamp = this.stampedLock.readLock();
        try {
            String string = String.format("ModernConcurrentMaterialStore[Size=%d, Types=%d, Value=$%.2f]", this.size(), this.groupByType().size(), this.getTotalInventoryValue());
            return string;
        }
        finally {
            this.stampedLock.unlockRead(stamp);
        }
    }
}

