package gameComponents;

import java.awt.Color;
import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Map;

/**
 * This utility class provides a number of methods that are useful for defining
 * and dealing with targets in the context of a shooter game.
 * 
 * The class encapsulates information about N different types of targets. Each
 * target type is identified by a unique index number, starting from 0 and
 * ranging up the the value N-1.
 * 
 * For each index, three attributes are defined: the points value of the target,
 * the colour of the target, and the size of the target. These three values
 * together can be used to create a specific kind of target.
 * 
 * @author mb
 * 
 */
public class TargetServices {

	private final static Map<Integer, String> colourNameMap = initColourNameMap();

	private final static Map<Integer, Integer> pointMap = initTargetPoints();
	private final static Map<Integer, Color> colourMap = initTargetColours();
	private final static Map<Integer, Integer> sizeMap = initTargetSizes();

	private TargetServices() {
		// we set the access modifier to "private" so that no client may invoke
		// this constructor (thereby ensuring that this class is a utility
		// class)
	}

	/**
	 * Returns the total number of targets defined by this utility class.
	 * 
	 * @return as described above
	 */
	public static int getNumberOfPossibleTargets() {
		return colourMap.size();
	}

	private static Map<Integer, String> initColourNameMap() {
		Map<Integer, String> map = new HashMap<Integer, String>();
		for (Field f : Color.class.getFields()) {
			if (f.getType() == Color.class) {
				Color c;
				try {
					c = (Color) f.get(null);
					map.put(c.getRGB(), f.getName());
					// System.out.println(c.getRGB() + "\t" + f.getName());
				} catch (IllegalArgumentException e) {
					e.printStackTrace();
				} catch (IllegalAccessException e) {
					e.printStackTrace();
				}
			}
		}
		return map;
	}

	private static Map<Integer, Integer> initTargetPoints() {
		Map<Integer, Integer> map = new HashMap<Integer, Integer>();
		map.put(0, 20);
		map.put(1, 30);
		map.put(2, 40);
		return map;
	}

	private static Map<Integer, Color> initTargetColours() {
		Map<Integer, Color> map = new HashMap<Integer, Color>();
		map.put(0, Color.BLUE);
		map.put(1, Color.RED);
		map.put(2, Color.MAGENTA);
		return map;
	}

	private static Map<Integer, Integer> initTargetSizes() {
		Map<Integer, Integer> map = new HashMap<Integer, Integer>();
		map.put(0, 50);
		map.put(1, 35);
		map.put(2, 20);
		return map;
	}

	/**
	 * Returns the text label for the given RGB integer value, if one exists. If
	 * a label for the passed RGB value is not defined in the standard palette,
	 * then this method returns null.
	 * 
	 * @return as described above
	 */
	public static String getColorName(int rgbValue) {
		return colourNameMap.get(rgbValue);
	}

	/**
	 * Returns the colour of the target type with the specified index number.
	 * 
	 * @param index
	 *            must be a value in the range [0,N-1] (where N is the total
	 *            number of targets defined)
	 * @return as described above
	 */
	public static Color getColorForIndex(int index) {
		return colourMap.get(index);
	}

	/**
	 * Returns the size attribute of the target type with the specified index
	 * number.
	 * 
	 * @param index
	 *            must be a value in the range [0,N-1] (where N is the total
	 *            number of targets defined)
	 * @return as described above
	 */
	public static Integer getSizeForIndex(int index) {
		return sizeMap.get(index);
	}

	/**
	 * Returns the points value of the target type with the specified index
	 * number.
	 * 
	 * @param index
	 *            must be a value in the range [0,N-1] (where N is the total
	 *            number of targets defined)
	 * @return as described above
	 */
	public static Integer getPointsForIndex(int index) {
		return pointMap.get(index);
	}

}
