package gameComponents;

import java.awt.Dimension;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Vector;

/**
 * This class encapsulates a data model for our game.
 * 
 * @author mb
 * 
 */
public class SpriteDataModel {

	private List<Sprite> theSprites;
	private List<ModelListener> listeners;

	private ShooterSprite theShooter;
	private ProjectileSprite theProjectile;
	private TargetSprite theTarget;
	private ScoreTallySprite theScore;

	private Dimension dimensionAvailable;

	/**
	 * Creates a collection and initialized it with the shooter sprite
	 */
	public SpriteDataModel(Dimension dim) {
		listeners = new Vector<ModelListener>();
		theSprites = new Vector<Sprite>();
		dimensionAvailable = dim;

		// here we create a sprite to represent our shooter
		// theShooter = new BasicTriangleShooter(dimensionAvailable);
		// EXERCISE #4 HERE IS THE OTHER CONSTRUCTOR
		theShooter = new FroggyShooter(dimensionAvailable);

		theShooter.setInitialLocation();

		// here we create a sprite to represent a target
		theTarget = new TargetSprite(dimensionAvailable);

		theScore = new ScoreTallySprite(dimensionAvailable);

		theSprites.add(theShooter);
		theSprites.add(theTarget);
		theSprites.add(theScore);
		notifyModelHasChanged();

	}

	public Dimension getDimension() {
		return dimensionAvailable;
	}

	/**
	 * Add a listener to this model.
	 * 
	 * @param listener
	 */
	public void addListener(ModelListener listener) {
		listeners.add(listener);
	}

	public void notifyModelHasChanged() {
		for (ModelListener listener : listeners) {
			listener.changed();
		}
	}

	/**
	 * Return this collection as a List of PolkaDot objects.
	 * 
	 * @return as described above
	 */
	public Collection<Sprite> getSprites() {
		return theSprites;
	}

	/**
	 * This method causes a firing behaviour to be shown on the display. A
	 * projectile is fired from the tip of the shooter.
	 */
	public void fireShooter() {
		theProjectile = theShooter.fire();
		theSprites.add(theProjectile);
		theScore.registerAmmoUsed();

		notifyModelHasChanged();
	}

	public ShooterSprite getShooterSprite() {
		return theShooter;
	}

	/**
	 * Move this sprite up one "vertical unit" (the size of this unit is
	 * determined by a class attribute).
	 */
	public void moveShooterUp() {
		getShooterSprite().moveUp();
		notifyModelHasChanged();
	}

	/**
	 * Move this sprite down one "vertical unit" (the size of this unit is
	 * determined by a class attribute).
	 */
	public void moveShooterDown() {
		getShooterSprite().moveDown();
		notifyModelHasChanged();
	}

	/**
	 * Move this sprite left one "horizontal unit" (the size of this unit is
	 * determined by a class attribute).
	 */
	public void moveShooterLeft() {
		getShooterSprite().moveLeft();
		notifyModelHasChanged();
	}

	/**
	 * Move this sprite right one "horizontal unit" (the size of this unit is
	 * determined by a class attribute).
	 */
	public void moveShooterRight() {
		getShooterSprite().moveRight();
		notifyModelHasChanged();
	}

	public void updateSprites() {
		boolean addNewTarget = false;
		Collection<ProjectileSprite> toBeRemoved = new LinkedList<ProjectileSprite>();

		for (Sprite s : getSprites()) {
			if (s != null && s instanceof ProjectileSprite) {
				ProjectileSprite ps = (ProjectileSprite) s;
				if (!ps.isAlive())
					toBeRemoved.add(ps);
				else {
					ps.move();
					if (theTarget.isCollided(s)) {
						theTarget.explode();
						theScore.registerHit(theTarget);
						addNewTarget = true;
					}
				}
			}
		}

		for (ProjectileSprite ps : toBeRemoved) {
			theSprites.remove(ps);
		}

		if (addNewTarget) {
			theTarget = new TargetSprite(dimensionAvailable);
			theSprites.add(theTarget);
		}
		notifyModelHasChanged();
	}

}
