package gameComponents;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.Toolkit;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

import javax.swing.ImageIcon;

/**
 * This class encapsulates an on-screen sprite that represents a shooter in our
 * asteroids-like video game.
 * 
 * This class supports linear translation of the sprite along the x and y
 * dimensions. It does not support rotation yet.
 * 
 * @author mb
 * 
 *         Version 2.0 of this class
 * 
 */
public class ScoreTallySprite implements Sprite {

	// the bounding box.
	private Rectangle2D boundingBox = null;
	// the anchor
	private Point2D anchor = new Point2D.Float(20, 20);
	private int width;
	private int height;
	private Font theFont = new Font("Serif", Font.BOLD, 14);

	private int score = 0;
	private int ammoTally = 0;

	private Map<String, Integer> scoreMap;

	private String prefix = "Score: ";

	/**
	 * Constructs a sprite located at the 0,0 position
	 * 
	 * @param dimensionOfDrawingRegion
	 */
	public ScoreTallySprite(Dimension dimensionOfDrawingRegion) {
		width = dimensionOfDrawingRegion.width;
		height = dimensionOfDrawingRegion.height;
		scoreMap = new TreeMap<String, Integer>();
	}

	/**
	 * Accessor for this sprite's current x position
	 * 
	 * @return as specified above
	 */
	public int getCurrentX() {
		return (int) getBounds2D().getX();
	}

	/**
	 * Accessor for this sprite's current y position
	 * 
	 * @return as specified above
	 */
	public int getCurrentY() {
		return (int) getBounds2D().getY();
	}

	/**
	 * This method takes the passed Graphics2D object and specifies how this
	 * sprite should be drawn on it.
	 */
	public void specifyDrawing(Graphics2D g) {

		String theText = prefix + score + "\n";
		
		// EXERCISE #2 - one statement solution is the line below
		theText = theText + "Ammo Tally: " + ammoTally + "\n";
		
		for (String t : scoreMap.keySet()) {
			theText += t + ": " + scoreMap.get(t) + "\n";
		}

		if (boundingBox == null) {
			FontMetrics metrics = g.getFontMetrics(theFont);
			// get the height of a line of text in this
			// font and render context
			int hgt = metrics.getHeight();
			// get the advance of my text in this font
			// and render context
			int adv = metrics.stringWidth(theText);
			// calculate the size of a box to hold the
			// text with some padding.
			boundingBox = new Rectangle2D.Double(anchor.getX(), anchor.getY(),
					adv + 2, hgt + 2);
		}

		// Render this sprite
		g.setColor(Color.RED);
		customDrawString(g, theText, (int) anchor.getX(), (int) anchor.getY());
	}

	/**
	 * Causes the passed string to be drawn on the provided Graphics2D object,
	 * anchored at the specified x,y coordinate. If the passed string contains
	 * newline characters, then the string will be drawn on multiple lines
	 * 
	 * @param g
	 * @param str
	 * @param x
	 * @param y
	 */
	private void customDrawString(Graphics2D g, String str, int x, int y) {
		for (String line : str.split("\n"))
			g.drawString(line, x, y += g.getFontMetrics().getHeight());
	}

	/**
	 * This method returns the width of the bounding box that contains this
	 * sprite
	 * 
	 * @return as described above
	 */
	public int getWidth() {
		return (int) getBounds2D().getWidth();
	}

	/**
	 * This method returns a rectangular box that completely encloses this
	 * sprite. The bounding box can be assumed to be the smallest bounding box
	 * that is possible.
	 * 
	 * The bounding box will be null until the string is rendered at least once.
	 * 
	 * @return as described above
	 */
	@Override
	public Rectangle2D getBounds2D() {
		return boundingBox;
	}

	/**
	 * This method returns the height of the bounding box that contains this
	 * sprite
	 * 
	 * @return as described above
	 */
	public int getHeight() {
		return (int) getBounds2D().getBounds2D().getHeight();
	}

	/**
	 * 
	 * 
	 * @param theTarget
	 */
	public void registerHit(TargetSprite theTarget) {
		score += theTarget.getPointsValue();
		Integer value = scoreMap.get(theTarget.toString());
		if (value == null) {
			scoreMap.put(theTarget.toString(), 1);
		} else {
			scoreMap.put(theTarget.toString(), value + 1);
		}
	}

	/**
	 * Update the culmulative ammo tally.
	 * 
	 */
	public void registerAmmoUsed() {
		ammoTally += 1;
	}

}
