package gameComponents;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Rectangle2D;

/**
 * This class encapsulates an on-screen sprite that represents a projectile in
 * our game.
 * 
 * A projectile is either live or dead. Only live projectiles are shown
 * on-screen. It has a linear trajectory that is represented by an x and y
 * displacement per frame.
 * 
 * @author mb
 * 
 */
public class ProjectileSprite implements Sprite {

	private Polygon theShape;
	private int worldWidth;
	private int worldHeight;
	private boolean isLive = false;

	private final int LIFETIME = 50;
	private int lifeTimeInFrames = LIFETIME;

	private int xDisplacement;
	private int yDisplacement;
	private double angularDisplacement;
	private boolean isFresh;

	/**
	 * Constructs a sprite located at the 0,0 position
	 * 
	 * @param dimensionOfDrawingRegion
	 */
	public ProjectileSprite(Dimension dimensionOfDrawingRegion,
			Point initialPosition, int xDisplacement, int yDisplacement) {

		worldWidth = dimensionOfDrawingRegion.width;
		worldHeight = dimensionOfDrawingRegion.height;
		this.xDisplacement = xDisplacement;
		this.yDisplacement = yDisplacement;
		if (yDisplacement < 0)
			this.angularDisplacement = Math.atan(-xDisplacement
					/ (double) yDisplacement);
		else
			this.angularDisplacement = Math.atan(-xDisplacement
					/ (double) yDisplacement)
					+ Math.PI;

		theShape = new Polygon();

		// specify the shape relative to the origin (upper left corner)

		int size = 6;

		theShape.addPoint(0, -0);
		theShape.addPoint(size, size);
		theShape.addPoint(-size, size);
		setAnchor(initialPosition.x, initialPosition.y);
		isLive = true;
		isFresh = true;

	}

	/**
	 * Mutates this sprite's anchor position
	 * 
	 * @param x
	 * @param y
	 */
	public void setAnchor(int x, int y) {
		theShape.translate(x, y);
	}

	/**
	 * Accessor for this sprite's current x position
	 * 
	 * @return as specified above
	 */
	public int getCurrentX() {
		return theShape.getBounds().x;
	}

	/**
	 * Accessor for this sprite's current y position
	 * 
	 * @return as specified above
	 */
	public int getCurrentY() {
		return theShape.getBounds().y;
	}

	public void initialPosition(Point p) {
		isLive = true;
	}

	public void move() {
		theShape.translate(xDisplacement, yDisplacement);

		// check if the projectile has moved off-screen
		boolean isOffScreenHorizontal = this.getCurrentX() > worldWidth
				|| this.getCurrentX() < 0;
		boolean isOffScreenVertical = this.getCurrentY() > worldHeight
				|| this.getCurrentY() < 0;
		if (isOffScreenHorizontal || isOffScreenVertical) {
			isLive = false;
		}

		// check if projectile's lifetime has elapsed
		if (lifeTimeInFrames == 0) {
			isLive = false;
		}

		// decrement this projectile's lifetime
		lifeTimeInFrames--;
		if (lifeTimeInFrames < LIFETIME / 2)
			isFresh = false;

	}

	/**
	 * This method takes the passed Graphics2D object and specifies how this
	 * sprite should be drawn on it.
	 */
	public void specifyDrawing(Graphics2D g) {

		if (isAlive()) {
			// Render this sprite
			if (isFresh)
				g.setColor(Color.RED);
			else
				g.setColor(Color.DARK_GRAY);

			g.rotate(angularDisplacement, theShape.getBounds2D().getCenterX(),
					theShape.getBounds2D().getCenterY());

			g.drawPolygon(theShape);
			g.fillPolygon(theShape);

			g.rotate(-angularDisplacement, theShape.getBounds2D().getCenterX(),
					theShape.getBounds2D().getCenterY());

		}
	}

	public boolean isAlive() {
		return isLive;
	}

	/**
	 * This method returns a rectangular box that completely encloses this
	 * sprite. The bounding box can be assumed to be the smallest bounding box
	 * that is possible.
	 * 
	 * @return as described above
	 */
	@Override
	public Rectangle2D getBounds2D() {
		return theShape.getBounds2D();
	}
}
