package gameComponents;

import java.awt.Canvas;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.geom.Dimension2D;
import java.awt.image.BufferStrategy;
import java.util.Collection;
import java.util.LinkedList;

/**
 * This class encapsulates a very simple window in which our shooter game can be
 * shown. This window provides the view of the game system. The view depends
 * upon the model in order to "know" which sprites are present.
 * 
 * At present, this class also encapsulates being an ActionListener and a
 * KeyListener. This window is capable of listening for events that indicate it
 * should be redrawn (in the case of multiple frames). This window is capable of
 * listening to events that are generated by the user pressing keys on the
 * keyboard. For these capabilities to actually be enacted, additional services
 * must used by the event dispatcher (to connect their ability to dispatch
 * events to this component's ability to listen)
 * 
 * @author mb
 * 
 */
public class GameCanvas extends Canvas implements ModelListener {

	private SpriteDataModel theModel;

	/**
	 * Construct a game canvas with the passed dimensions
	 */
	public GameCanvas(Dimension dim, SpriteDataModel theModel) {
		this.setMinimumSize(dim);
		this.theModel = theModel;

		// here we install this view as a listener on the model.
		// this makes sense because this view depends upon the model in order to
		// know how to draw itself.
		theModel.addListener(this);
	}

	/**
	 * This method accomplishes the painting of this canvas.
	 * 
	 * This method is called when the contents of the component should be
	 * painted; such as when the component is first being shown or is damaged
	 * and in need of repair.
	 * 
	 * It should be understood that the main method of an app typically does not
	 * directly invoke this method. Rather, the application indicates that the
	 * canvas is in need of painting via the invalidate() method, which signals
	 * that the canvas needs to be redrawn. For instance, when the game state
	 * changes (like the user has moved a sprite or fired a shot or whatever).
	 * The window manager, once alerted to the fact that a window needs to be
	 * redrawn, will trigger a process that will ultimately result in this
	 * method being invoked.
	 */
	public void paint(Graphics g) {
		Graphics2D g2 = (Graphics2D) g;
		drawFrame(g2);
	}

	/**
	 * This method implements the functionality so that the scene is drawn on
	 * the back buffer, and then the entire back buffer is shown at once.
	 * 
	 * This method accomplishes a performance improvement. Instead of the
	 * graphics being drawn repeatedly, for each time an on-screen element is
	 * specified (incrementally), the buffer is rendered all at once.
	 * 
	 * Implementation note: this method delegates to another method the
	 * specification of what the screen elements are.
	 * 
	 * @param g
	 *            the Graphics2D object that is encapsulating the graphics
	 *            context on which the on-screen elements are being drawn.
	 */
	public void drawFrame(Graphics2D g) {

		BufferStrategy bufStrategy = this.getBufferStrategy();
		Graphics2D backBuffer = (Graphics2D) bufStrategy.getDrawGraphics();

		// clear the backbuffer
		backBuffer.setPaint(Color.WHITE);
		backBuffer.fillRect(0, 0, getWidth(), getHeight());

		// here we draw our game world (nicely delegated to another method)
		drawOnScreenElements(backBuffer);

		// Clean up
		backBuffer.dispose();
		bufStrategy.show();
		Toolkit.getDefaultToolkit().sync();
	}

	/**
	 * This method draws all of the required on-screen elements on the passed
	 * graphics context.
	 * 
	 * Implementation note: the client should not assume that the graphics
	 * context corresponds to what is shown on-screen. In fact, it may be a
	 * graphics buffer that will be rendered at some other point.
	 * 
	 * @param g
	 *            as specified above
	 */
	public void drawOnScreenElements(Graphics2D g) {
		for (Sprite s : theModel.getSprites()) {
			if (s != null) {
				s.specifyDrawing(g);
			}
		}
	}

	/**
	 * The method alerts this view that the model upon which it depends has
	 * changed. As well, a statement is printed to the console
	 * "View detects that model has changed and signals it needs to be redrawn."
	 */
	@Override
	public void changed() {
		// System.out
		// .println("View detects that model has changed and signals it needs to be redrawn.");
		this.repaint();
	}

}
