package gameComponents;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.Toolkit;
import java.awt.geom.AffineTransform;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import javax.swing.ImageIcon;

/**
 * This class encapsulates an on-screen sprite that represents a shooter in our
 * asteroids-like video game.
 * 
 * This class supports linear translation of the sprite along the x and y
 * dimensions. It does not support rotation yet.
 * 
 * @author mb
 * 
 *         Version 2.0 of this class
 * 
 */
public class FroggyShooter implements ShooterSprite {

	private Image img;

	private Polygon theShape;
	/*
	 * tip of the shooter, expressed as a displacement from the anchor point of
	 * the bounding box.
	 */
	private Point tipRelativeToCentre;

	private Dimension availableRegion;

	/*
	 * Number of coordinate units moved in the horizontal direction for any
	 * single left or right movement
	 */
	private int X_DELTA = 10;

	/*
	 * Number of coordinate units moved in the vertical direction for any single
	 * up or down movement
	 */
	private int Y_DELTA = 10;

	private double angularDisplacement = 0;

	private final double ANGLE_DELTA = Math.PI / 45;

	/**
	 * Constructs a sprite located at the 0,0 position
	 * 
	 * @param dimensionOfDrawingRegion
	 */
	public FroggyShooter(Dimension dimensionOfDrawingRegion) {
		this.availableRegion = dimensionOfDrawingRegion;

		img = Toolkit.getDefaultToolkit().getImage("assets/frog3.jpg");

		ImageIcon icon = new ImageIcon(img);
		int width = (int) (icon.getIconWidth() * 0.75);
		int height = (int) (icon.getIconHeight() * 0.75);
		tipRelativeToCentre = new Point((int) (-0.06 * width),
				(int) (-0.34 * height));

		// centre the shape around the point that is below the tip
		theShape = new Polygon();
		theShape.addPoint(0, 0);
		theShape.addPoint(width, 0);
		theShape.addPoint(width, height);
		theShape.addPoint(0, height);

	}

	private void setAnchor(int x, int y) {
		int transX = x - getCurrentX();
		int transY = y - getCurrentY();
		theShape.translate(transX, transY);
	}

	private int getCurrentX() {
		return (int) getBounds2D().getX();
	}

	private int getCurrentY() {
		return (int) getBounds2D().getY();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gameComponents.ShooterSpriteI#specifyDrawing(java.awt.Graphics2D)
	 */
	@Override
	public void specifyDrawing(Graphics2D g) {
		g.rotate(angularDisplacement, theShape.getBounds2D().getCenterX(),
				theShape.getBounds2D().getCenterY());
		g.drawImage(img, getCurrentX(), getCurrentY(), getWidth(), getHeight(),
				null);
		g.rotate(-angularDisplacement, theShape.getBounds2D().getCenterX(),
				theShape.getBounds2D().getCenterY());
	}

	private int getWidth() {
		return (int) theShape.getBounds2D().getWidth();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gameComponents.ShooterSpriteI#getBounds2D()
	 */
	@Override
	public Rectangle2D getBounds2D() {
		return theShape.getBounds2D();
	}

	private int getHeight() {
		return (int) theShape.getBounds2D().getHeight();
	}

	private Point getShooterTip() {

		Point bboxPoint = boundingBoxBoundaryPointRelativeToCentre();

		UtilityCircle circ = new UtilityCircle(bboxPoint);

		// now use same angle, but smaller diameter
		double tipDist = Math.sqrt(Math.pow(tipRelativeToCentre.getX(), 2)
				+ Math.pow(tipRelativeToCentre.getY(), 2));

		UtilityCircle circ3 = new UtilityCircle(angularDisplacement
				- circ.getAngle(), tipDist);
		double xDisplacement = circ3.getPointOnCircumference().getX();
		double yDisplacement = circ3.getPointOnCircumference().getY();

		return new Point(
				(int) (theShape.getBounds2D().getCenterX() + xDisplacement),
				(int) (theShape.getBounds2D().getCenterY() + yDisplacement));

	}

	/**
	 * This method returns a point on the bounding box defined as follows.
	 * Suppose a line is drawn from the centre of the bounding box, through the
	 * shooterAnchor point and reaching to the edge of the bounding box. The
	 * point at which the line reaches the bouding box is the point returned by
	 * this method
	 * 
	 * @return
	 */
	private Point boundingBoxBoundaryPointRelativeToCentre() {
		double yCoord = theShape.getBounds2D().getHeight() / 2;
		yCoord = -yCoord;
		// following derivation is based on isolating x in expression: y = mx+b,
		// and using known values:
		// b=0, m=(y2-y1)/(x2-x1). The 2 coords are: (0,0) and
		// (tipRelativeToCentre.getX(),
		// tipRelativeToCentre.getY())
		double xCoord = yCoord * tipRelativeToCentre.getX()
				/ tipRelativeToCentre.getY();
		return new Point((int) xCoord, (int) yCoord);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gameComponents.ShooterSpriteI#moveUp()
	 */
	@Override
	public void moveUp() {
		boolean willGoOffScreen = theShape.getBounds2D().getY() + Y_DELTA < 0;
		if (!willGoOffScreen) {
			theShape.translate(0, -Y_DELTA);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gameComponents.ShooterSpriteI#moveDown()
	 */
	@Override
	public void moveDown() {
		boolean willGoOffScreen = theShape.getBounds2D().getY()
				+ theShape.getBounds2D().getHeight() + Y_DELTA >= availableRegion
				.getHeight();
		if (!willGoOffScreen) {
			theShape.translate(0, Y_DELTA);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gameComponents.ShooterSpriteI#moveLeft()
	 */
	@Override
	public void moveLeft() {
		boolean willGoOffScreen = theShape.getBounds2D().getX() - X_DELTA < 0;
		if (!willGoOffScreen) {
			theShape.translate(-X_DELTA, 0);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gameComponents.ShooterSpriteI#moveRight()
	 */
	@Override
	public void moveRight() {
		boolean willGoOffScreen = theShape.getBounds2D().getX()
				+ theShape.getBounds2D().getWidth() + X_DELTA >= availableRegion
				.getWidth();
		if (!willGoOffScreen) {
			theShape.translate(X_DELTA, 0);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gameComponents.ShooterSpriteI#rotateLeft()
	 */
	@Override
	public void rotateLeft() {
		angularDisplacement = angularDisplacement + ANGLE_DELTA;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gameComponents.ShooterSpriteI#rotateLeft()
	 */
	@Override
	public void rotateRight() {
		angularDisplacement = angularDisplacement - ANGLE_DELTA;
	}

	@Override
	public void setInitialLocation() {
		// here we position the sprite at the bottom centre, above a certain
		// PADDING
		final int BOTTOM_PADDING = 20;

		this.setAnchor(
				(int) (availableRegion.getWidth() - this.getWidth()) / 2,
				(int) availableRegion.getHeight() - this.getHeight()
						- BOTTOM_PADDING);
	}

	@Override
	public ProjectileSprite fire() {
		int hypotenuse = 10;
		int yDisplacement = (int) (Math.cos(angularDisplacement) * hypotenuse);
		int xDisplacement = (int) (Math.sin(angularDisplacement) * hypotenuse);

		return new ProjectileSprite(availableRegion, this.getShooterTip(),
				xDisplacement, -yDisplacement);
	}

}
