package gameComponents;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.Toolkit;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import javax.swing.ImageIcon;

/**
 * This class encapsulates an on-screen sprite that represents a shooter in our
 * asteroids-like video game.
 * 
 * This class supports linear translation of the sprite along the x and y
 * dimensions. It does not support rotation yet.
 * 
 * @author mb
 * 
 *         Version 2.0 of this class
 * 
 */
public class BasicTriangleShooter implements ShooterSprite {

	private Polygon theShape;

	private double angularDisplacement = 0;

	/*
	 * tip of the shooter, expressed as a displacement from the anchor point of
	 * the bounding box.
	 */
	private Point tipRelativeToAnchor;

	private Dimension availableRegion;

	/*
	 * Number of coordinate units moved in the horizontal direction for any
	 * single left or right movement
	 */
	private int X_DELTA = 10;

	/*
	 * Number of coordinate units moved in the vertical direction for any single
	 * up or down movement
	 */
	private int Y_DELTA = 10;

	private final double ANGLE_DELTA = Math.PI / 45;

	/**
	 * Constructs a sprite located at the 0,0 position
	 * 
	 * @param dimensionOfDrawingRegion
	 */
	public BasicTriangleShooter(Dimension dimensionOfDrawingRegion) {
		this.availableRegion = dimensionOfDrawingRegion;

		theShape = new Polygon();

		// specify the shape, which will be a triangle with its apex set to
		// (0,0). The apex of the triangle corresponds to the tip of the
		// shooter.
		int widthOfBase = 60;
		int heightOfTriangle = 20;
		theShape.addPoint(0, 0);
		// bottom left and right corners
		theShape.addPoint(widthOfBase / 2, heightOfTriangle);
		theShape.addPoint(-widthOfBase / 2, heightOfTriangle);
		// now the bounding box of the triangle will be anchored at (-30,0) and
		// will have width and height of widthOfBase and heightOfTriangle,
		// respectively.
		// Thus, the apex of the triangle will be located at (0,0). The
		// displacement from the anchor to the tip is thus 30 units in the x
		// dimension and 0 units in the vertical dimension
		tipRelativeToAnchor = new Point(30, 0);

	}

	private void setAnchor(int x, int y) {
		int transX = x - getCurrentX();
		int transY = y - getCurrentY();
		theShape.translate(transX, transY);
	}

	private int getCurrentX() {
		return (int) getBounds2D().getX();
	}

	private int getCurrentY() {
		return (int) getBounds2D().getY();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gameComponents.ShooterSpriteI#specifyDrawing(java.awt.Graphics2D)
	 */
	@Override
	public void specifyDrawing(Graphics2D g) {
		// Render this sprite
		g.setColor(Color.RED);

		g.rotate(angularDisplacement, theShape.getBounds2D().getCenterX(),
				theShape.getBounds2D().getCenterY());

		g.drawPolygon(theShape);

		g.rotate(-angularDisplacement, theShape.getBounds2D().getCenterX(),
				theShape.getBounds2D().getCenterY());
	}

	private int getWidth() {
		return (int) theShape.getBounds2D().getWidth();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gameComponents.ShooterSpriteI#getBounds2D()
	 */
	@Override
	public Rectangle2D getBounds2D() {
		return theShape.getBounds2D();
	}

	private int getHeight() {
		return (int) theShape.getBounds2D().getHeight();
	}

	/**
	 */
	private Point getShooterTip() {
		return new Point(
				(int) (theShape.getBounds2D().getX() + tipRelativeToAnchor
						.getX()),
				(int) (theShape.getBounds2D().getY() + tipRelativeToAnchor
						.getY()));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gameComponents.ShooterSpriteI#moveUp()
	 */
	@Override
	public void moveUp() {
		boolean willGoOffScreen = theShape.getBounds2D().getY() + Y_DELTA < 0;
		if (!willGoOffScreen) {
			theShape.translate(0, -Y_DELTA);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gameComponents.ShooterSpriteI#moveDown()
	 */
	@Override
	public void moveDown() {
		boolean willGoOffScreen = theShape.getBounds2D().getY()
				+ theShape.getBounds2D().getHeight() + Y_DELTA >= availableRegion
				.getHeight();
		if (!willGoOffScreen) {
			theShape.translate(0, Y_DELTA);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gameComponents.ShooterSpriteI#moveLeft()
	 */
	@Override
	public void moveLeft() {
		boolean willGoOffScreen = theShape.getBounds2D().getX() - X_DELTA < 0;
		// if (!willGoOffScreen) {
		// theShape.translate(-X_DELTA, 0);
		// }

		// ## EXERCISE #1B
		// to implement wrapping, use the same test. If test is true, then add
		// screen width to
		int xDisp = -X_DELTA;
		if (willGoOffScreen) {
			xDisp = availableRegion.width - X_DELTA;
		}
		theShape.translate(xDisp, 0);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gameComponents.ShooterSpriteI#moveRight()
	 */
	@Override
	public void moveRight() {
		// boolean willGoOffScreen = theShape.getBounds2D().getX()
		// + theShape.getBounds2D().getWidth() + X_DELTA >= availableRegion
		// .getWidth();

		boolean willGoOffScreen = theShape.getBounds2D().getX() + X_DELTA >= availableRegion
				.getWidth();

		// if (!willGoOffScreen) {
		// theShape.translate(X_DELTA, 0);
		// }

		// ## EXERCISE #1B
		// to implement wrapping, use the same test. If test is true, then
		// subtract
		// screen width to translation amount

		// modify the test to remove the width of the shooter shape from the
		// test (do this as a separate step)

		int xDisp = X_DELTA;
		if (willGoOffScreen) {
			xDisp = -availableRegion.width + X_DELTA;
		}
		theShape.translate(xDisp, 0);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see gameComponents.ShooterSpriteI#rotateLeft()
	 */
	@Override
	public void rotateLeft() {
		angularDisplacement = angularDisplacement + ANGLE_DELTA;
	}

	@Override
	public void setInitialLocation() {
		// here we position the sprite at the bottom centre, above a certain
		// PADDING
		final int BOTTOM_PADDING = 20;

		this.setAnchor(
				(int) (availableRegion.getWidth() - this.getWidth()) / 2,
				(int) availableRegion.getHeight() - this.getHeight()
						- BOTTOM_PADDING);
	}

	@Override
	public ProjectileSprite fire() {

		int hypotenuse = 10;
		int yDisplacement = (int) (Math.cos(angularDisplacement) * hypotenuse);
		int xDisplacement = (int) (Math.sin(angularDisplacement) * hypotenuse);

		return new ProjectileSprite(availableRegion, this.getShooterTip(),
				xDisplacement, -yDisplacement);
	}

	@Override
	public void rotateRight() {
		angularDisplacement = angularDisplacement - ANGLE_DELTA;
	}

}
