package utility;

import java.awt.Point;

/**
 * This class encapsulates a generator for a trajectory for a particle. Its
 * primary service is provided via the getNextPoint() method.
 * 
 * @author mb
 * 
 */
public class TrajectoryGenerator {
	private final int xMax;
	private final int yMax;
	private Point currPoint;
	private final int MAX_DISPLACEMENT = 5;

	/**
	 * Construct an instance of a generator of a trajectory. The generated
	 * trajectory will not venture out of the bounds passed to this constructor.
	 * 
	 * @param horizontalDimension
	 *            a non-zero positive value
	 * @param verticalDimension
	 *            a non-zero positive value
	 */
	public TrajectoryGenerator(int horizontalDimension, int verticalDimension) {
		xMax = horizontalDimension;
		yMax = verticalDimension;
		currPoint = new Point(xMax / 2, yMax / 2);
	}

	/**
	 * This method returns the next point of this particle's trajectory. It is
	 * randomly chosen from among a finite set of possible next moves in any
	 * possible direction. The next point will not venture out of bounds.
	 * 
	 * @return as described above.
	 */
	public Point getNextPoint() {
		int newX = currPoint.x
				+ getRandomIntFromClosedInterval(-MAX_DISPLACEMENT,
						MAX_DISPLACEMENT);
		boolean isXWithinBounds = newX < xMax && newX > 0;
		// for no bounds checking, use the following condition instead
		// boolean isXWithinBounds = true;

		while (!isXWithinBounds) {
			newX = currPoint.x
					+ getRandomIntFromClosedInterval(-MAX_DISPLACEMENT,
							MAX_DISPLACEMENT);
			isXWithinBounds = newX < xMax && newX > 0;
		}

		int newY = currPoint.y
				+ getRandomIntFromClosedInterval(-MAX_DISPLACEMENT,
						MAX_DISPLACEMENT);
		boolean isYWithinBounds = newY < yMax && newY > 0;
		// for no bounds checking, use the following condition instead
		// boolean isYWithinBounds = true;
		while (!isYWithinBounds) {
			newY = currPoint.y
					+ getRandomIntFromClosedInterval(-MAX_DISPLACEMENT,
							MAX_DISPLACEMENT);
			isYWithinBounds = newY < yMax && newY > 0;
		}

		currPoint = new Point(newX, newY);
		return currPoint;
	}

	private static int getRandomIntFromClosedInterval(int minDesiredVal,
			int maxDesiredVal) {
		int randomVal = minDesiredVal
				+ (int) (Math.random() * (maxDesiredVal - minDesiredVal + 1));
		return randomVal;
	}

}
