package helper;

import java.text.DecimalFormat;

/**
 * This class encapsulates a cargo container that can hold boxes of stereos and boxes
 * of TV (no other type of cargo). It provides a limited variety of services,
 * such as adding to and modifying the contents of the cargo container.
 * 
 */
public class CargoContainer {

	private int boxesOfTVs;
	private int boxesOfStereos;

	/**
	 * A field that determines the cost of TV
	 **/
	private final double PRICE_TV = 200;
	/**
	 * A field that determines the cost stereo
	 **/
	private final double PRICE_STEREO = 100;

	/**
	 * A flag that determines whether the return of toProperString is the number
	 * of boxesOfStereos and the number boxesOfTVs or the monetary value of the
	 * contents of cargo container. The default value is true.
	 **/
	private boolean asValue = true;

	/**
	 * Constructs a default cargo container with the value of boxesOfTVs as 0 and
	 * the number of boxesOfStereos as 0.
	 **/
	public CargoContainer() {
		super();
		this.boxesOfTVs = 0;
		this.boxesOfStereos = 0;
	}

	/**
	 * Constructs a cargo container with the passed value of boxesOfTVs and the
	 * passed value of boxesOfStereos. TV and boxesOfStereos cannot be negative.
	 * 
	 * @throws Exception
	 *             if the number of boxes of stereos or the number of boxes of TVs is
	 *             less than zero
	 **/
	public CargoContainer(int boxesOfStereos, int boxesOfTVs) {
		super();
		crashStereos(boxesOfStereos < 0);
		crashTV(boxesOfTVs < 0);

		this.boxesOfTVs = boxesOfTVs;
		this.boxesOfStereos = boxesOfStereos;
	}

	/**
	 * Add the passed value of boxesOfStereos to the cargo container
	 * 
	 * @throws Exception
	 *             if the number of boxes of stereos is less than zero
	 * 
	 **/
	public void addBoxesOfStereos(int boxesOfStereos) {
		crashStereos(boxesOfStereos < 0);
		this.boxesOfStereos = this.boxesOfStereos + boxesOfStereos;
	}

	/**
	 * Add the passed value of boxesOfTVs to the cargo container
	 * 
	 * @throws Exception
	 *             if the number of boxes of TVs is less than zero
	 **/
	public void addBoxesOfTVs(int boxesOfTVs) {
		crashTV(boxesOfTVs < 0);
		this.boxesOfTVs = this.boxesOfTVs + boxesOfTVs;
	}

	/** Returns the number of boxesOfTVs in the cargo container **/
	public int getBoxesOfTVs() {
		return boxesOfTVs;
	}

	/**
	 * Sets the number of boxes of TVs in this cargo container to be the passed
	 * quantity.
	 * 
	 * @param boxesOfTVs
	 *            the number of boxes of TVs that should be in this cargo
	 *            container
	 * 
	 * @throws Exception
	 *             if the number of boxes of TVs is less than zero
	 */
	public void setBoxesOfTVs(int boxesOfTVs) {
		crashTV(boxesOfTVs < 0);
		this.boxesOfTVs = boxesOfTVs;
	}

	/** Returns the number of boxesOfStereos in the cargo container **/
	public int getBoxesOfStereos() {
		return boxesOfStereos;
	}

	/**
	 * Sets the number of boxes of stereos in this cargo container to be the passed
	 * quantity.
	 * 
	 * @param boxesOfStereos
	 *            the number of boxes of stereos that should be in this cargo
	 *            container
	 * 
	 * @throws Exception
	 *             if the passed quantity is less than zero
	 */
	public void setBoxesOfStereos(int boxesOfStereos) {
		crashStereos(boxesOfStereos < 0);
		this.boxesOfStereos = boxesOfStereos;
	}

	/**
	 * Returns the value of the boxes of stereos contained in this cargo container
	 * in dollars
	 **/
	public double getValueOfBoxesOfStereos() {
		return boxesOfStereos * PRICE_STEREO;
	}

	/**
	 * Returns the value of the boxes of TVs contained in this cargo container in
	 * dollars
	 **/
	public double getValueOfBoxesOfTVs() {
		return boxesOfTVs * PRICE_TV;
	}

	/**
	 * Returns the value of the contents of this cargo container (defined as all
	 * the boxes of stereos and boxes of TVs) in dollars
	 **/
	public double getTotalValueOfCargoContainer() {
		return getValueOfBoxesOfStereos() + getValueOfBoxesOfTVs();
	}

	/**
	 * Adds the contents of the passed cargo container to the contents of this
	 * cargo container.
	 **/
	public void mergeContents(CargoContainer otherCargoContainer) {
		this.addBoxesOfStereos(otherCargoContainer.getBoxesOfStereos());
		this.addBoxesOfTVs(otherCargoContainer.getBoxesOfTVs());

	}

	/**
	 * Checks whether this cargo container is equal to the passed vending
	 * machine, where two machines are considered equal if their contents have
	 * the same number of boxes of TVs and the same number of boxes of stereos.
	 **/
	@Override
	public boolean equals(Object otherCargoContainer) {
		if (this == otherCargoContainer)
			return true;
		if (otherCargoContainer == null)
			return false;
		if (getClass() != otherCargoContainer.getClass())
			return false;
		CargoContainer other = (CargoContainer) otherCargoContainer;
		if (boxesOfStereos != other.boxesOfStereos)
			return false;
		if (boxesOfTVs != other.boxesOfTVs)
			return false;
		return true;
	}

	/**
	 * Sets whether this object, when printed out using its toProperString()
	 * method, should shows itself as the monetary value of its contents or as a
	 * formatted string that lists the contents of this cargo container. By default
	 * it is set to true (and toProperString displays the monetary value).
	 **/
	public void shouldShowValue(boolean shouldShowValue) {
		this.asValue = shouldShowValue;
	}

	/**
	 * Checks whether the value of the contents of this cargo container is equal
	 * to the value of the contents of the passed cargo container
	 **/
	public boolean valueEquals(CargoContainer otherCargoContainer) {

		return this.getTotalValueOfCargoContainer() == otherCargoContainer
				.getTotalValueOfCargoContainer();
	}

	/**
	 * Returns a string representation of this cargo container that is either
	 * the monetary value of its contents or a formatted description of the
	 * contents of this cargo container (in terms of the number of boxes of stereos
	 * and boxes of TVs contained in this cargo container). Which of these two
	 * options is used depends on this objects state with respect to
	 * shouldShowValue (see the method shoudShowValue(boolean). The default
	 * option is the first one.
	 **/
	public String toProperString() {
		DecimalFormat df = new DecimalFormat("0.00");
		return asValue ? "$"
				+ df.format((double) getTotalValueOfCargoContainer())
				: getBoxesOfStereos() + " boxes of stereos , and " + getBoxesOfTVs()
						+ " boxes of TVs";
	}

	private void crashTV(boolean negTV) {
		if (negTV)
			try {
				throw new Exception("number of boxesOfTVs cannot be negative");
			} catch (Exception e) {
				e.printStackTrace();

			}

	}

	private void crashStereos(boolean negStereos) {
		if (negStereos)
			try {
				throw new Exception("number of boxesOfStereos cannot be negative");
			} catch (Exception e) {
				e.printStackTrace();
			}
	}

}
