package helper;

import java.text.DecimalFormat;

/**
 * This class encapsulates a vending bank that can hold bags of chips and cans
 * of pop (no other type of food). It provides a limited variety of services,
 * such as adding to and modifying the contents of the vending machine.
 * 
 */
public class VendingMachine {

	private int cansOfPop;
	private int bagsOfChips;

	/**
	 * A field that determines the cost of pop
	 **/
	private final double PRICE_POP = 2;
	/**
	 * A field that determines the cost chips
	 **/
	private final double PRICE_CHIPS = 1;

	/**
	 * A flag that determines whether the return of toProperString is the number
	 * of bagsOfChips and the number bagsOfChips or the monetary value of the
	 * contents of vending machine. The default value is true.
	 **/
	private boolean asValue = true;

	/**
	 * Constructs a default vending machine with the value of cansOfPop as 0 and
	 * the number of bagsOfChips as 0.
	 **/
	public VendingMachine() {
		super();
		this.cansOfPop = 0;
		this.bagsOfChips = 0;
	}

	/**
	 * Constructs a vending machine with the passed value of cansOfPop and the
	 * passed value of bagsOfChips. Pop and bagsOfChips cannot be negative.
	 * 
	 * @throws Exception
	 *             if the number of bags of chip or the number of cans of pop is
	 *             less than zero
	 **/
	public VendingMachine(int bagsOfChips, int cansOfPop) {
		super();
		crashChips(bagsOfChips < 0);
		crashPop(cansOfPop < 0);

		this.cansOfPop = cansOfPop;
		this.bagsOfChips = bagsOfChips;
	}

	/**
	 * Add the passed value of bagsOfChips to the vending machine
	 * 
	 * @throws Exception
	 *             if the number of bags of chip is less than zero
	 * 
	 **/
	public void addBagsOfChips(int bagsOfChips) {
		crashChips(bagsOfChips < 0);
		this.bagsOfChips = this.bagsOfChips + bagsOfChips;
	}

	/**
	 * Add the passed value of cansOfPop to the vending machine
	 * 
	 * @throws Exception
	 *             if the number of cans of pop is less than zero
	 **/
	public void addCansOfPop(int cansOfPop) {
		crashPop(cansOfPop < 0);
		this.cansOfPop = this.cansOfPop + cansOfPop;
	}

	/** Returns the number of cansOfPop in the vending machine **/
	public int getCansOfPop() {
		return cansOfPop;
	}

	/**
	 * Sets the number of cans of pop in this vending machine to be the passed
	 * quantity.
	 * 
	 * @param cansOfPop
	 *            the number of cans of pop that should be in this vending
	 *            machine
	 * 
	 * @throws Exception
	 *             if the number of cans of pop is less than zero
	 */
	public void setCansOfPop(int cansOfPop) {
		crashPop(cansOfPop < 0);
		this.cansOfPop = cansOfPop;
	}

	/** Returns the number of bagsOfChips in the vending machine **/
	public int getBagsOfChips() {
		return bagsOfChips;
	}

	/**
	 * Sets the number of bags of chips in this vending machine to be the passed
	 * quantity.
	 * 
	 * @param bagsOfChips
	 *            the number of bags of chips that should be in this vending
	 *            machine
	 * 
	 * @throws Exception
	 *             if the passed quantity is less than zero
	 */
	public void setBagsOfChips(int bagsOfChips) {
		crashChips(bagsOfChips < 0);
		this.bagsOfChips = bagsOfChips;
	}

	/**
	 * Returns the value of the bags of chips contained in this vending machine
	 * in dollars
	 **/
	public double getValueOfBagsOfChips() {
		return bagsOfChips * PRICE_CHIPS;
	}

	/**
	 * Returns the value of the cans of pop contained in this vending machine in
	 * dollars
	 **/
	public double getValueOfCansOfPop() {
		return cansOfPop * PRICE_POP;
	}

	/**
	 * Returns the value of the contents of this vending machine (defined as all
	 * the bags of chips and cans of pop) in dollars
	 **/
	public double getTotalValueOfVendingMachine() {
		return getValueOfBagsOfChips() + getValueOfCansOfPop();
	}

	/**
	 * Adds the contents of the passed vending machine to the contents of this
	 * vending machine.
	 **/
	public void mergeContents(VendingMachine otherVendingmachine) {
		this.addBagsOfChips(otherVendingmachine.getBagsOfChips());
		this.addCansOfPop(otherVendingmachine.getCansOfPop());

	}

	/**
	 * Checks whether this vending machine is equal to the passed vending
	 * machine, where two machines are considered equal if their contents have
	 * the same number of cans of pop and the same number of bags of chips.
	 **/
	@Override
	public boolean equals(Object otherVendingmachine) {
		if (this == otherVendingmachine)
			return true;
		if (otherVendingmachine == null)
			return false;
		if (getClass() != otherVendingmachine.getClass())
			return false;
		VendingMachine other = (VendingMachine) otherVendingmachine;
		if (bagsOfChips != other.bagsOfChips)
			return false;
		if (cansOfPop != other.cansOfPop)
			return false;
		return true;
	}

	/**
	 * Sets whether this object, when printed out using its toProperString()
	 * method, should shows itself as the monetary value of its contents or as a
	 * formatted string that lists the contents of this vending machine.
	 **/
	public void shouldShowValue(boolean shouldShowValue) {
		this.asValue = shouldShowValue;
	}

	/**
	 * Checks whether the value of the contents of this vending machine is equal
	 * to the value of the contents of the passed vending machine
	 **/
	public boolean valueEquals(VendingMachine otherVendingmachine) {

		return this.getTotalValueOfVendingMachine() == otherVendingmachine
				.getTotalValueOfVendingMachine();
	}

	/**
	 * Returns a string representation of this vending machine that is either
	 * the monetary value of its contents or a formatted description of the
	 * contents of this vending machine (in terms of the number of bags of chips
	 * and cans of pop contained in this vending machine). Which of these two
	 * options is used depends on this objects state with respect to
	 * shouldShowValue (see the method shoudShowValue(boolean). The default
	 * option is the first one.
	 **/
	public String toProperString() {
		DecimalFormat df = new DecimalFormat("0.00");
		return asValue ? "$"
				+ df.format((double) getTotalValueOfVendingMachine())
				: getBagsOfChips() + " bags of chips , and " + getCansOfPop()
						+ " cans of pop";
	}

	private void crashPop(boolean negPop) {
		if (negPop)
			try {
				throw new Exception("number of cansOfPop cannot be negative");
			} catch (Exception e) {
				e.printStackTrace();

			}

	}

	private void crashChips(boolean negChips) {
		if (negChips)
			try {
				throw new Exception("number of bagsOfChips cannot be negative");
			} catch (Exception e) {
				e.printStackTrace();
			}
	}

}
