package helper;

import java.text.DecimalFormat;

/**
 * This class encapsulates a piggy bank that can hold quarters and nickels (no
 * other coins). It provides a limited variety of services, such as adding to
 * and modifying the piggy bank.
 * 
 */
public class Piggybank {

	private int quarters;
	private int nickels;

	/**
	 * A flag that determines whether the return of toProperString is the number
	 * of nickels and the number nickels or the monetary value of the piggy
	 * bank. The default value is true.
	 **/
	private boolean asValue = true;

	/**
	 * Constructs a default piggy bank with the value of quarters as 0 and the
	 * number of nickels as 0.
	 **/
	public Piggybank() {
		super();
		this.quarters = 0;
		this.nickels = 0;
	}

	/**
	 * Constructs a piggy bank with the passed value of quarters and the passed
	 * value of nickels. Quarters and nickels cannot be negative.
	 **/
	public Piggybank(int nickels, int quarters) {
		super();
		crashNickels(nickels < 0);
		crashQuarters(quarters < 0);

		this.quarters = quarters;
		this.nickels = nickels;
	}

	/** Add the passed value of nickels to the piggy bank **/
	public void addNickels(int nickels) {
		crashNickels(nickels < 0);
		this.nickels = this.nickels + nickels;
	}

	/** Add the passed value of quarters to the piggy bank **/
	public void addQuarters(int quarters) {
		crashQuarters(quarters < 0);
		this.quarters = this.quarters + quarters;
	}

	/** Returns the number of quarters in the piggy bank **/
	public int getQuarters() {
		return quarters;
	}

	/**
	 * Sets the number of quarters in this piggybank to be the passed quantity.
	 * 
	 * @param quarters
	 *            the number of quarters that should be in this piggybank
	 * 
	 * @throws Exception
	 *             if the number of quarters is less than zerio
	 */
	public void setQuarters(int quarters) {
		crashQuarters(quarters < 0);
		this.quarters = quarters;
	}

	/** Returns the number of nickels in the piggy bank **/
	public int getNickels() {
		return nickels;
	}

	/**
	 * Sets the number of nickels in this piggybank to be the passed quantity.
	 * 
	 * @param nickels
	 *            the number of nickels that should be in this piggybank
	 * 
	 * @throws Exception
	 *             if the number of nickels is less than zerio
	 */
	public void setNickels(int nickels) {
		crashNickels(nickels < 0);
		this.nickels = nickels;
	}

	/** Returns the value of the nickels in the piggybank in cents **/
	public int getValueOfNickels() {
		return nickels * 5;
	}

	/** Returns the value of the quarters in the piggybank in cents **/
	public int getValueOfQuarters() {
		return quarters * 25;
	}

	public int getTotalValueOfPiggybank() {
		return getValueOfNickels() + getValueOfQuarters();
	}

	/** Adds the coins in the passed piggy bank to the coins in this piggybank **/
	public void mergePiggybanks(Piggybank otherPiggybank) {
		this.addNickels(otherPiggybank.getNickels());
		this.addQuarters(otherPiggybank.getQuarters());

	}

	/** Checks whether this piggy bank is equal to the passed piggy bank **/
	@Override
	public boolean equals(Object otherPiggybank) {
		if (this == otherPiggybank)
			return true;
		if (otherPiggybank == null)
			return false;
		if (getClass() != otherPiggybank.getClass())
			return false;
		Piggybank other = (Piggybank) otherPiggybank;
		if (nickels != other.nickels)
			return false;
		if (quarters != other.quarters)
			return false;
		return true;
	}

	/**
	 * Checks whether the value of the piggy bank is equal to the passed piggy
	 * bank
	 **/
	public boolean hasSameValue(Piggybank otherPiggybank) {

		return this.getTotalValueOfPiggybank() == otherPiggybank
				.getTotalValueOfPiggybank();
	}

	/**
	 * Sets whether this object, when printed out using its toProperString()
	 * method, shows itself as a monetary value or as a formatted string that
	 * lists the contents of this piggybank.
	 **/
	public void shouldShowValue(boolean shouldShowValue) {
		this.asValue = shouldShowValue;
	}

	/**
	 * Returns a string representation of this Piggy Bank that is either its
	 * monetary value or a formatted description of the contents of this
	 * piggybank (in terms of the number of quarters and nickels contained in
	 * this piggy bank). Which of these two options is used depends on this
	 * objects state with respect to shouldShowValue (see the method
	 * shoudShowValue(boolean). The default option is the first one.
	 **/
	public String toProperString() {
		DecimalFormat df = new DecimalFormat("0.00");
		return asValue ? "$"
				+ df.format((double) getTotalValueOfPiggybank() / 100)
				: getNickels() + " nickels , and " + getQuarters()
						+ " quarters";
	}

	private void crashQuarters(boolean negQuarters) {
		if (negQuarters)
			try {
				throw new Exception("quarters cannot be negative");
			} catch (Exception e) {
				e.printStackTrace();

			}

	}

	private void crashNickels(boolean negNickels) {
		if (negNickels)
			try {
				throw new Exception("nickels cannot be negative");
			} catch (Exception e) {
				e.printStackTrace();
			}
	}

}
